import os
import json
import time
import gspread
from google.oauth2.service_account import Credentials
from amazon.paapi import AmazonAPI
from datetime import datetime, timedelta
import re
import unicodedata
import paramiko

# Konfiguration importieren
from config.PSW_config import (
    GOOGLE_SHEET_ID, GOOGLE_SHEET_TAB, GOOGLE_CREDENTIALS_FILE,
    AMAZON_ACCESS_KEY, AMAZON_SECRET_KEY, AMAZON_ASSOCIATE_TAG,
    AMAZON_COUNTRY, SFTP_HOST, SFTP_PORT, SFTP_USER, SFTP_PASS, SFTP_TARGET_PATH
)

# --- Google Sheets einlesen ---
scopes = ['https://www.googleapis.com/auth/spreadsheets']
creds = Credentials.from_service_account_file(GOOGLE_CREDENTIALS_FILE, scopes=scopes)
gc = gspread.authorize(creds)
sheet = gc.open_by_key(GOOGLE_SHEET_ID).worksheet(GOOGLE_SHEET_TAB)
rows = sheet.get_all_records()

# --- Amazon API Setup ---
amazon = AmazonAPI(
    AMAZON_ACCESS_KEY,
    AMAZON_SECRET_KEY,
    AMAZON_ASSOCIATE_TAG,
    AMAZON_COUNTRY
)

def slugify(text):
    if not text:
        return ""
    # Umlaute ersetzen
    text = text.replace("ä", "ae").replace("ö", "oe").replace("ü", "ue").replace("ß", "ss")
    # Unicode-Normalisierung (z.B. é → e)
    text = unicodedata.normalize('NFKD', text).encode('ascii', 'ignore').decode('ascii')
    # Alles außer Buchstaben/Zahlen durch "-" ersetzen
    text = re.sub(r'[^a-zA-Z0-9]+', '-', text)
    # Zu Kleinbuchstaben, führende/abschließende - entfernen
    return text.strip('-').lower()

# Spaltenindizes finden oder erstellen
def get_column_index(sheet, column_name):
    """Findet den Index einer Spalte oder erstellt sie, falls nicht vorhanden"""
    header_row = sheet.row_values(1)
    
    if column_name in header_row:
        return header_row.index(column_name) + 1  # gspread verwendet 1-basierte Indizes
    else:
        # Spalte hinzufügen
        col_index = len(header_row) + 1
        sheet.update_cell(1, col_index, column_name)
        return col_index

# Indizes für die Spalten finden/erstellen
try:
    asin_col = get_column_index(sheet, 'ASIN')
    shop_col = get_column_index(sheet, 'Shop')
    titel_col = get_column_index(sheet, 'Titel')
    beschreibung_col = get_column_index(sheet, 'Kurzbeschreibung')
    kategorie_col = get_column_index(sheet, 'Kategorie')
    produkt_url_col = get_column_index(sheet, 'Produkt_URL')
    affiliate_url_col = get_column_index(sheet, 'affiliate_url')
    bild_url_col = get_column_index(sheet, 'bild_url')
    update_col = get_column_index(sheet, 'letztes_update')
except Exception as e:
    print(f"Fehler beim Ermitteln der Spaltenindizes: {e}")
    exit()

# Funktion zur Überprüfung, ob ein Update notwendig ist
def needs_update(last_update_str, min_days=30):
    """Prüft, ob das letzte Update mehr als min_days Tage zurückliegt"""
    if not last_update_str:
        return True
    
    try:
        last_update = datetime.strptime(last_update_str, "%Y-%m-%d %H:%M")
        return datetime.now() - last_update > timedelta(days=min_days)
    except ValueError:
        return True

# ASINs identifizieren, die aktualisiert werden müssen
asins_to_update = []
result_list = []

for i, row in enumerate(rows, start=2):  # start=2 wegen Header-Zeile
    asin = row.get('ASIN', '')
    shop = row.get('Shop', '')
    last_update = row.get('letztes_update', '')

    eigenschaften_raw = row.get('Eigenschaften', '') or row.get('Eigenschaft', '')

    try:
        eigenschaften = json.loads(eigenschaften_raw) if eigenschaften_raw else {}
    except Exception as e:
        print(f"Eigenschaften-Parsing-Fehler bei {row.get('ID')}: {e}")
        eigenschaften = {}
    
    # Alle Produkte in die JSON aufnehmen
    produkt = {
        "id": row.get('ID', ''),
        "asin": asin,
        "shop": shop,
        "titel": row.get('Titel', ''),
        "slug": slugify(row.get('Titel', '')),
        "beschreibung": row.get('Kurzbeschreibung', ''),
        "kategorie": row.get('Kategorie', ''),
        "eigenschaften": eigenschaften,    # <--- NEU!
        "affiliate_url": row.get('affiliate_url', ''),
        "bild_url": row.get('bild_url', ''),
        "letztes_update": last_update
    }
    result_list.append(produkt)
    
    # Nur Amazon-Produkte mit ASIN und veraltetem Update aktualisieren
    if shop.lower() == 'amazon' and asin and needs_update(last_update):
        asins_to_update.append((asin, i))  # (ASIN, Zeilenindex)

# --- Amazon-Produkte aktualisieren ---
if asins_to_update:
    print(f"Aktualisiere {len(asins_to_update)} Amazon-Produkte...")
    
    batch_size = 10
    current_time = time.strftime("%Y-%m-%d %H:%M")
    
    for i in range(0, len(asins_to_update), batch_size):
        batch = asins_to_update[i:i+batch_size]
        asin_batch = [item[0] for item in batch]
        
        try:
            response = amazon.get_items(asin_batch)
            data = response['data']
            
            for asin, row_index in batch:
                item = data.get(asin)
                if not item:
                    print(f"Keine Daten für ASIN {asin} gefunden")
                    continue
                
                # Bild-URL extrahieren
                bild_url = (
                    item.images.primary.large.url
                    if hasattr(item, "images")
                    and item.images
                    and hasattr(item.images, "primary")
                    and item.images.primary
                    and hasattr(item.images.primary, "large")
                    and item.images.primary.large
                    and hasattr(item.images.primary.large, "url")
                    else ""
                )
                
                # Affiliate-URL extrahieren
                affiliate_url = item.detail_page_url if hasattr(item, "detail_page_url") else ""
                
                # Werte in die Tabelle schreiben
                sheet.update_cell(row_index, affiliate_url_col, affiliate_url)
                sheet.update_cell(row_index, bild_url_col, bild_url)
                sheet.update_cell(row_index, update_col, current_time)
                
                # Auch in der result_list aktualisieren
                for produkt in result_list:
                    if produkt['asin'] == asin:
                        produkt['affiliate_url'] = affiliate_url
                        produkt['bild_url'] = bild_url
                        produkt['letztes_update'] = current_time
                        break
                        
        except Exception as e:
            print(f"Fehler bei ASINs {asin_batch}: {e}")
else:
    print("Keine Amazon-Produkte zur Aktualisierung gefunden.")

# --- Lokale JSON schreiben ---
local_json = "PSW_produkte.json"
with open(local_json, 'w', encoding='utf-8') as f:
    json.dump(result_list, f, ensure_ascii=False, indent=2)
print(f"Lokale Datei geschrieben: {local_json}")

# --- SFTP-Upload ---
def upload_via_sftp(local_file, remote_file, host, port, user, password):
    transport = paramiko.Transport((host, port))
    transport.connect(username=user, password=password)
    sftp = paramiko.SFTPClient.from_transport(transport)
    try:
        sftp.put(local_file, remote_file)
        print(f"SFTP-Upload erfolgreich: {remote_file}")
    except Exception as e:
        print(f"Fehler beim SFTP-Upload: {e}")
    finally:
        sftp.close()
        transport.close()

upload_via_sftp(
    local_file=local_json,
    remote_file=SFTP_TARGET_PATH,
    host=SFTP_HOST,
    port=SFTP_PORT,
    user=SFTP_USER,
    password=SFTP_PASS
)

print("Skript erfolgreich abgeschlossen.")