<?php
/*
Plugin Name: Amazon Affiliate Klick-Tracking
Description: DSGVO-konformes Klick-Tracking für Amazon Affiliate Links mit einfacher Dateispeicherung
Version: 1.1
Author: Dein Name
Text Domain: amazon-affiliate-tracking
*/

// Sicherstellen, dass das Plugin nicht direkt aufgerufen wird
if (!defined('ABSPATH')) {
    exit;
}

class AmazonAffiliateTracking {
    
    private $data_file;
    
    public function __construct() {
        // Pfad zur JSON-Datei in uploads
        $upload_dir = wp_upload_dir();
        $this->data_file = $upload_dir['basedir'] . '/amazon_clicks.json';
        
        // Hook für die Installation
        register_activation_hook(__FILE__, array($this, 'install'));
        
        // Hooks für Admin-Menü
        add_action('admin_menu', array($this, 'add_admin_menu'));
        
        // Hook für Tracking (wird von redirect-affbox.php aufgerufen) - JETZT MIT 3 PARAMETERN!
        add_action('amazon_track_click', array($this, 'track_click'), 10, 3);
    }
    
    // Initialisierung
    public function install() {
        // Stelle sicher, dass die Datei existiert
        if (!file_exists($this->data_file)) {
            $this->save_data(array());
        }
    }
    
    // Daten speichern
    private function save_data($data) {
        file_put_contents($this->data_file, json_encode($data, JSON_PRETTY_PRINT));
    }
    
    // Daten laden
    private function load_data() {
        if (!file_exists($this->data_file)) {
            return array();
        }
        
        $data = file_get_contents($this->data_file);
        return json_decode($data, true) ?: array();
    }
    
    // Klick tracken mit Seiten-Information
    public function track_click($produkt, $slug, $source_url = '') {
        $data = $this->load_data();
        $current_date = current_time('Y-m-d');
        
        $produkt_id = $produkt['id'];
        
        // Finde oder erstelle Eintrag für dieses Produkt
        if (!isset($data[$produkt_id])) {
            $data[$produkt_id] = array(
                'slug' => $slug,
                'clicks' => array(),
                'sources' => array() // Neue Quelle-Statistik
            );
        }
        
        // Erhöhe Klick-Zähler für heutiges Datum
        if (!isset($data[$produkt_id]['clicks'][$current_date])) {
            $data[$produkt_id]['clicks'][$current_date] = 0;
        }
        $data[$produkt_id]['clicks'][$current_date]++;
        
        // Tracke Quelle (Beitragsseite) - NUR wenn übergeben
        if (!empty($source_url)) {
            if (!isset($data[$produkt_id]['sources'][$source_url])) {
                $data[$produkt_id]['sources'][$source_url] = 0;
            }
            $data[$produkt_id]['sources'][$source_url]++;
        }
        
        // Speichere Daten
        $this->save_data($data);
        
        return true;
    }
    
    // Admin-Menü hinzufügen
    public function add_admin_menu() {
        add_submenu_page(
            'tools.php',
            'Amazon Klick-Statistiken',
            'Amazon Klicks',
            'manage_options',
            'amazon-click-stats',
            array($this, 'display_stats_page')
        );
    }
    
    // Statistiken anzeigen
    public function display_stats_page() {
        $data = $this->load_data();
        
        // Zeitraum-Filter
        $time_filter = isset($_GET['time_filter']) ? sanitize_text_field($_GET['time_filter']) : '30days';
        
        // Berechne Gesamtstatistiken
        $stats = $this->calculate_stats($data, $time_filter);
        
        ?>
        <div class="wrap">
            <h1>Amazon Affiliate Klick-Statistiken</h1>
            
            <!-- Zeitraum-Filter -->
            <div style="margin: 20px 0;">
                <form method="get">
                    <input type="hidden" name="page" value="amazon-click-stats">
                    <label for="time_filter">Zeitraum:</label>
                    <select name="time_filter" id="time_filter" onchange="this.form.submit()">
                        <option value="7days" <?php selected($time_filter, '7days'); ?>>Letzte 7 Tage</option>
                        <option value="30days" <?php selected($time_filter, '30days'); ?>>Letzte 30 Tage</option>
                        <option value='90days' <?php selected($time_filter, '90days'); ?>>Letzte 90 Tage</option>
                        <option value='year' <?php selected($time_filter, 'year'); ?>>Letztes Jahr</option>
                        <option value='all' <?php selected($time_filter, 'all'); ?>>Alle Zeit</option>
                    </select>
                </form>
            </div>
            
            <!-- Übersichts-Karten -->
            <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(250px, 1fr)); gap: 20px; margin-bottom: 30px;">
                <div style="background: #fff; padding: 20px; border-radius: 8px; box-shadow: 0 2px 5px rgba(0,0,0,0.1);">
                    <h3 style="margin-top: 0; color: #2271b1;">Gesamtklicks</h3>
                    <p style="font-size: 2em; font-weight: bold; color: #2271b1;"><?php echo number_format($stats['total_clicks']); ?></p>
                </div>
                
                <div style="background: #fff; padding: 20px; border-radius: 8px; box-shadow: 0 2px 5px rgba(0,0,0,0.1);">
                    <h3 style="margin-top: 0; color: #00a32a;">Heute</h3>
                    <p style="font-size: 2em; font-weight: bold; color: #00a32a;"><?php echo number_format($stats['today_clicks']); ?></p>
                </div>
                
                <div style="background: #fff; padding: 20px; border-radius: 8px; box-shadow: 0 2px 5px rgba(0,0,0,0.1);">
                    <h3 style="margin-top: 0; color: #d63638;">Gestern</h3>
                    <p style="font-size: 2em; font-weight: bold; color: #d63638;"><?php echo number_format($stats['yesterday_clicks']); ?></p>
                </div>
            </div>
            
            <!-- Top Produkte -->
            <div style="background: #fff; padding: 20px; border-radius: 8px; box-shadow: 0 2px 5px rgba(0,0,0,0.1); margin-bottom: 30px;">
                <h2>Top Produkte</h2>
                <table class="widefat fixed" cellspacing="0">
                    <thead>
                        <tr>
                            <th>Produkt ID</th>
                            <th>Slug</th>
                            <th>Klicks</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php if (!empty($stats['top_products'])) : ?>
                            <?php foreach ($stats['top_products'] as $product_id => $product_data) : ?>
                                <tr>
                                    <td><?php echo esc_html($product_id); ?></td>
                                    <td><?php echo esc_html($product_data['slug']); ?></td>
                                    <td><?php echo number_format($product_data['total_clicks']); ?></td>
                                </tr>
                            <?php endforeach; ?>
                        <?php else : ?>
                            <tr>
                                <td colspan="3" style="text-align: center;">Keine Klicks gefunden</td>
                            </tr>
                        <?php endif; ?>
                    </tbody>
                </table>
            </div>
            
            <!-- Tägliche Statistik -->
            <div style="background: #fff; padding: 20px; border-radius: 8px; box-shadow: 0 2px 5px rgba(0,0,0,0.1); margin-bottom: 30px;">
                <h2>Klicks der letzten 30 Tage</h2>
                <table class="widefat fixed" cellspacing="0">
                    <thead>
                        <tr>
                            <th>Datum</th>
                            <th>Klicks</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php if (!empty($stats['daily_clicks'])) : ?>
                            <?php foreach ($stats['daily_clicks'] as $date => $clicks) : ?>
                                <tr>
                                    <td><?php echo date('d.m.Y', strtotime($date)); ?></td>
                                    <td><?php echo number_format($clicks); ?></td>
                                </tr>
                            <?php endforeach; ?>
                        <?php else : ?>
                            <tr>
                                <td colspan="2" style="text-align: center;">Keine Daten verfügbar</td>
                            </tr>
                        <?php endif; ?>
                    </tbody>
                </table>
            </div>
            
            <!-- Detaillierte Produkt-Statistiken -->
            <div style="background: #fff; padding: 20px; border-radius: 8px; box-shadow: 0 2px 5px rgba(0,0,0,0.1); margin-bottom: 30px;">
                <h2>Detaillierte Produkt-Statistiken</h2>
                <?php foreach ($data as $product_id => $product_data) : ?>
                    <div style="margin-bottom: 30px; padding: 15px; border: 1px solid #ddd;">
                        <h3>Produkt: <?php echo esc_html($product_id); ?> (<?php echo esc_html($product_data['slug']); ?>)</h3>
                        <p><strong>Gesamtklicks:</strong> <?php echo number_format(array_sum($product_data['clicks'])); ?></p>
                        <?php $this->display_source_stats($product_id, $product_data); ?>
                    </div>
                <?php endforeach; ?>
            </div>
            
        </div>
        <?php
    }
    
    // Statistiken berechnen
    private function calculate_stats($data, $time_filter) {
        $stats = array(
            'total_clicks' => 0,
            'today_clicks' => 0,
            'yesterday_clicks' => 0,
            'top_products' => array(),
            'daily_clicks' => array()
        );
        
        $current_date = current_time('Y-m-d');
        $yesterday_date = date('Y-m-d', strtotime('-1 day'));
        
        // Zeitliche Begrenzung basierend auf Filter
        $cutoff_date = null;
        switch ($time_filter) {
            case '7days':
                $cutoff_date = date('Y-m-d', strtotime('-7 days'));
                break;
            case '30days':
                $cutoff_date = date('Y-m-d', strtotime('-30 days'));
                break;
            case '90days':
                $cutoff_date = date('Y-m-d', strtotime('-90 days'));
                break;
            case 'year':
                $cutoff_date = date('Y-m-d', strtotime('-1 year'));
                break;
        }
        
        foreach ($data as $product_id => $product_data) {
            $product_total = 0;
            
            foreach ($product_data['clicks'] as $date => $clicks) {
                // Überspringe Daten außerhalb des gewählten Zeitraums
                if ($cutoff_date && $date < $cutoff_date && $time_filter !== 'all') {
                    continue;
                }
                
                $product_total += $clicks;
                $stats['total_clicks'] += $clicks;
                
                // Tägliche Statistik
                if (!isset($stats['daily_clicks'][$date])) {
                    $stats['daily_clicks'][$date] = 0;
                }
                $stats['daily_clicks'][$date] += $clicks;
                
                // Heute und Gestern
                if ($date === $current_date) {
                    $stats['today_clicks'] += $clicks;
                }
                if ($date === $yesterday_date) {
                    $stats['yesterday_clicks'] += $clicks;
                }
            }
            
            if ($product_total > 0) {
                $stats['top_products'][$product_id] = array(
                    'slug' => $product_data['slug'],
                    'total_clicks' => $product_total
                );
            }
        }
        
        // Sortiere Top-Produkte nach Klicks
        uasort($stats['top_products'], function($a, $b) {
            return $b['total_clicks'] - $a['total_clicks'];
        });
        
        // Sortiere tägliche Klicks nach Datum (neueste zuerst)
        krsort($stats['daily_clicks']);
        
        return $stats;
    }
    
    // Statistik-Funktionen für externe Nutzung
    public static function get_product_clicks($produkt_id) {
        $instance = new self();
        $data = $instance->load_data();
        
        if (isset($data[$produkt_id])) {
            return array_sum($data[$produkt_id]['clicks']);
        }
        
        return 0;
    }
    
    public static function get_total_clicks() {
        $instance = new self();
        $data = $instance->load_data();
        $total = 0;
        
        foreach ($data as $product_data) {
            $total += array_sum($product_data['clicks']);
        }
        
        return $total;
    }
    
    // Quelle-Statistik anzeigen
    private function display_source_stats($product_id, $product_data) {
        if (empty($product_data['sources'])) {
            echo '<p>Keine Quellen-Informationen verfügbar</p>';
            return;
        }
        
        // Sortiere nach meisten Klicks
        arsort($product_data['sources']);
        ?>
        <div style="margin-top: 20px;">
            <h4>Klick-Quellen für dieses Produkt</h4>
            <table class="widefat fixed" cellspacing="0" style="margin-top: 10px;">
                <thead>
                    <tr>
                        <th>Beitragsseite</th>
                        <th>Klicks</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($product_data['sources'] as $source => $clicks) : ?>
                        <tr>
                            <td>
                                <a href="<?php echo esc_url($source); ?>" target="_blank">
                                    <?php echo esc_html($this->shorten_url($source)); ?>
                                </a>
                            </td>
                            <td><?php echo number_format($clicks); ?></td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
        <?php
    }

    // URL für Anzeige kürzen
    private function shorten_url($url, $length = 60) {
        if (strlen($url) <= $length) {
            return $url;
        }
        return substr($url, 0, $length) . '...';
    }
}

// Plugin initialisieren
new AmazonAffiliateTracking();

// Hilfsfunktionen für externe Nutzung
function amazon_get_product_clicks($produkt_id) {
    return AmazonAffiliateTracking::get_product_clicks($produkt_id);
}

function amazon_get_total_clicks() {
    return AmazonAffiliateTracking::get_total_clicks();
}
